/*
* Creation date : Mon Mar 05 15:13:57 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
* CE2_RSA_BIULD functions.
*
* \version CE2_RSA_BUILD.c#1:csrc:1
* \author Pavel Sasunkevich
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "CE2_RSA_BUILD.h"
#include "LLF_RSA_BUILD.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name:
*  CE2_RSA_Build_PubKey
*
* Inputs:
*  @param UserPubKey_ptr [in/out] - A pointer to the public key structure;
*  @param Exponent_ptr [in] - A pointer to the exponent stream of bytes;
*  @param ExponentSize [in] - The size of the exponent, in bytes;
*  @param Modulus_ptr  [in] - A pointer to the modulus stream of bytes;
*  @param ModulusSize  [in] - The size of the modulus, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Build_PubKey populates a CE2RSAPubKey_t structure with
*  the provided modulus and exponent.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to build public key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Build_PubKey(
                        CE2_RSAUserPubKey_t *UserPubKey_ptr,/* out */
                        DxUint8_t *Exponent_ptr,			/* in */
                        DxUint16_t ExponentSize,			/* in */
                        DxUint8_t *Modulus_ptr,				/* in */
                        DxUint16_t ModulusSize )			/* in */
{
	if (UserPubKey_ptr == DX_NULL)
		return CE2_RSA_BUILD_PUBKEY_INVALID_PUB_KEY_PTR_ERROR;

	if (Exponent_ptr == DX_NULL)
		return CE2_RSA_BUILD_PUBKEY_INVALID_EXPONENT_PTR_ERROR;

	if (Modulus_ptr == DX_NULL)
		return CE2_RSA_BUILD_PUBKEY_INVALID_MODULUS_PTR_ERROR;

	if (((*Modulus_ptr) >> 7) != 0x01)
		return CE2_RSA_BUILD_PUBKEY_INVALID_MODULUS_MS_BIT_ERROR;

	if ((CE2_RSA_MIN_VALID_KEY_SIZE_VALUE_IN_BITS >> 3) > ModulusSize ||
        ModulusSize > (CE2_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS >> 3) ||
		(ModulusSize % (CE2_RSA_VALID_KEY_SIZE_MULTIPLE_VALUE_IN_BITS >> 3) != 0))
		return CE2_RSA_BUILD_PUBKEY_INVALID_MODULUS_SIZE;

	return LLF_RSA_Build_PubKey(UserPubKey_ptr, Exponent_ptr,
		ExponentSize, Modulus_ptr, ModulusSize );
} /* End of CE2_RSA_Build_PubKey */

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Build_PrivKey
*
* Inputs:
*  @param UserPrivKey_ptr [out] - A pointer to the public key structure.
*  @param PrivExponent_ptr [in] - A pointer to the private exponent stream of bytes;
*  @param PrivExponentSize [in] - The size of the private exponent, in bytes;
*  @param PubExponent_ptr  [in]- A pointer to the public exponent stream of bytes;
*  @param PubExponentSize  [in]- The size of the public exponent, in bytes;
*  @param Modulus_ptr      [in]- A pointer to the modulus stream of bytes.
*  @param ModulusSize      [in]- The size of the modulus, in bytes. 
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Build_PrivKey populates a CE2RSAPrivKey_t structure with
* the provided modulus and exponent, marking the key as a "non-CRT" key
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PrivKey
*     to build private key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Build_PrivKey(
                        CE2_RSAUserPrivKey_t   *UserPrivKey_ptr,	/* out */
	                    DxUint8_t              *PrivExponent_ptr,	/* in */
                        DxUint16_t              PrivExponentSize,	/* in */
                        DxUint8_t              *PubExponent_ptr,	/* in */
                        DxUint16_t              PubExponentSize,	/* in */
                        DxUint8_t              *Modulus_ptr,		/* in */
                        DxUint16_t              ModulusSize )		/* in */
{
	if (UserPrivKey_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEY_INVALID_PRIV_KEY_PTR_ERROR;

	if (PrivExponent_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEY_INVALID_PRIV_EXPONENT_PTR_ERROR;

	if (PubExponent_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEY_INVALID_PUB_EXPONENT_PTR_ERROR;

	if (Modulus_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEY_INVALID_MODULUS_PTR_ERROR;

	if (((*Modulus_ptr) >> 7) != 0x01)
		return CE2_RSA_BUILD_PRIVKEY_INVALID_MODULUS_MS_BIT_ERROR;

	if ((CE2_RSA_MIN_VALID_KEY_SIZE_VALUE_IN_BITS >> 3) > ModulusSize ||
		ModulusSize > (CE2_RSA_MAX_VALID_KEY_SIZE_VALUE_IN_BITS >> 3) ||
		(ModulusSize % (CE2_RSA_VALID_KEY_SIZE_MULTIPLE_VALUE_IN_BITS >> 3) != 0))
		return CE2_RSA_BUILD_PRIVKEY_INVALID_MODULUS_SIZE;

	return LLF_RSA_Build_PrivKey(UserPrivKey_ptr, PrivExponent_ptr, PrivExponentSize,
		PubExponent_ptr, PubExponentSize, Modulus_ptr, ModulusSize);
} /* End of CE2_RSA_Build_PrivKey */

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Build_PrivKeyCRT
*
* Inputs:
*  @param UserPrivKey_ptr [in/out] - A pointer to the user private  key structure.
*  @param P_ptr [out] - a pointer to the first factor stream of bytes;
*  @param PSize [in/out] - the size of the first factor buffer in bytes, it is
*         updated to the actual size of the first factor, in bytes;
*  @param Q_ptr [out] - a pointer to the second factor stream of bytes;
*  @param QSize [in/out] - the size of the second factor buffer in bytes , it is
*         updated to the actual size of the second factor, in bytes;
*  @param dP_ptr [out] - a pointer to the first factors CRT exponent stream of bytes;
*  @param dPSize [in/out] - the size of the first factor exponent buffer in bytes, it is
*         updated to the actual size of the first factor exponent, in bytes;
*  @param dQ_ptr [out] - a pointer to the second factors CRT exponent stream of bytes;
*  @param dQSize [in/out] - the size of the second factors CRT exponent buffer in bytes, it is
*         updated to the actual size of the second factors CRT exponent, in bytes;
*  @param qInv_ptr [out] - a pointer to the first CRT coefficient stream of bytes;
*  @param qInvSize [in/out] - the size of the first CRT coefficient buffer in bytes, it is
*         updated to the actual size of the first CRT coefficient, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Build_PrivKeyCRT populates a CE2RSAPrivKey_t structure with
*  the provided parameters, marking the key as a "CRT" key.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PrivKeyCRT
*     to build private key with CRT values with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Build_PrivKeyCRT(
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,	/* out */
                        DxUint8_t *P_ptr, 	/* in */
                        DxUint16_t PSize,	/* in */
                        DxUint8_t *Q_ptr,	/* in */
                        DxUint16_t QSize,	/* in */
                        DxUint8_t *dP_ptr, /* in */
                        DxUint16_t dPSize,	/* in */
                        DxUint8_t *dQ_ptr,	/* in */
                        DxUint16_t dQSize,	/* in */
                        DxUint8_t *qInv_ptr,	/* in */
                        DxUint16_t qInvSize )	/* in */
{
	if (UserPrivKey_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEYCRT_INVALID_PRIV_KEY_PTR_ERROR;

	if (P_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEYCRT_INVALID_P_PTR_ERROR;

	if (Q_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEYCRT_INVALID_Q_PTR_ERROR;

	if (dP_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEYCRT_INVALID_DP_PTR_ERROR;

	if (dQ_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEYCRT_INVALID_DQ_PTR_ERROR;

	if (qInv_ptr == DX_NULL)
		return CE2_RSA_BUILD_PRIVKEYCRT_INVALID_QINV_PTR_ERROR;

	return LLF_RSA_Build_PrivKeyCRT(UserPrivKey_ptr, P_ptr, PSize, Q_ptr, QSize,
		dP_ptr, dPSize, dQ_ptr, dQSize, qInv_ptr, qInvSize);
} /* End of CE2_RSA_Build_PrivKeyCRT */

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_PubKey
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key structure. 
*  @param Exponent_ptr   [out] - A pointer to the exponent stream of bytes
*  @param ExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes. 
*  @param Modulus_ptr    [in] - A pointer to the modulus stream of bytes.
*  @param ModulusSize_ptr [in/out] - the size of the modulus buffer in bytes,
*         it is updated to the actual size of the modulus, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Get_PubKey gets the e,n public key from the database.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to get from public key public exponent (e) and modulus (N)
*     with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_PubKey(
                        CE2_RSAUserPubKey_t *UserPubKey_ptr,	/* in */
                        DxUint8_t  *Exponent_ptr,				/* out */
                        DxUint16_t *ExponentSize_ptr,			/* in, out */
                        DxUint8_t  *Modulus_ptr,				/* out */
                        DxUint16_t *ModulusSize_ptr )			/* in, out */
{
	if (UserPubKey_ptr == DX_NULL)
		return CE2_RSA_GET_PUBKEY_INVALID_PUB_KEY_PTR_ERROR;

	if (Exponent_ptr == DX_NULL)
		return CE2_RSA_GET_PUBKEY_INVALID_EXPONENT_PTR_ERROR;

	if (Modulus_ptr == DX_NULL)
		return CE2_RSA_GET_PUBKEY_INVALID_MODULUS_PTR_ERROR;

	return LLF_RSA_Get_PubKey(UserPubKey_ptr, Exponent_ptr, ExponentSize_ptr,
		Modulus_ptr, ModulusSize_ptr);
} /* End of CE2_RSA_Get_PubKey */

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_ModSizeFromPubKey
*
* Inputs:
*  @param UserPubKey_ptr [in] - A pointer to the public key structure,
*         as returned by CE2_RSA_Build_PubKey.
*  @param ModulusSize_ptr [out] -  The actual size of the modulus, in bytes;
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*   The function gets modulus size from public key.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Get_ModSizeFromPubKey
*     to get modulus size from public key with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_ModSizeFromPubKey(
                        CE2_RSAUserPubKey_t *UserPubKey_ptr,/* in */
                        DxUint16_t *ModulusSize_ptr )		/* out */
{
	if (UserPubKey_ptr == DX_NULL)
		return CE2_RSA_GET_MODSIZEFROMPUBKEY_INVALID_PUB_KEY_PTR_ERROR;

	if (ModulusSize_ptr == DX_NULL)
		return CE2_RSA_GET_MODSIZEFROMPUBKEY_INVALID_MODULUS_PTR_ERROR;

	return LLF_RSA_Get_ModSizeFromPubKey(UserPubKey_ptr, ModulusSize_ptr);
} /* End of CE2_RSA_Get_ModSizeFromPubKey */

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_PrivKey
*
* Inputs:
*  @param UserPrivKey_ptr [in] - A pointer to the private key structure;
*  @param PrivExponent_ptr [out] - A pointer to the exponent stream of bytes;
*  @param PrivExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param PubExponent_ptr [out] - a pointer to the public exponent stream of bytes;
*  @param PubExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param Modulus_ptr     [out] - A pointer to the modulus stream of bytes.
*  @param ModulusSize_ptr [in/out] - the size of the modolous buffer in bytes,
*         it is updated to the actual size of the modolous, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Get_PrivKey gets the D, n private key from the database.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to get from private key public exponent (e), private exponent
*     (d) and modulus (N) with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_PrivKey(
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,		/* in */
                        DxUint8_t             *PrivExponent_ptr,	/* out */
                        DxUint16_t            *PrivExponentSize_ptr,/* in, out */
                        DxUint8_t             *PubExponent_ptr,		/* out */
                        DxUint16_t            *PubExponentSize_ptr,	/* in, out */
                        DxUint8_t             *Modulus_ptr,			/* out */
                        DxUint16_t            *ModulusSize_ptr )	/* in, out */
{
	if (UserPrivKey_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEY_INVALID_PRIV_KEY_PTR_ERROR;

	if (PrivExponent_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEY_INVALID_PRIV_EXPONENT_PTR_ERROR;

	if (PrivExponentSize_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEY_INVALID_PRIV_EXPONENT_SIZE_PTR_ERROR;

	if (PubExponent_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEY_INVALID_PUB_EXPONENT_PTR_ERROR;

	if (PubExponentSize_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEY_INVALID_PUB_EXPONENT_SIZE_PTR_ERROR;

	if (Modulus_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEY_INVALID_MODULUS_PTR_ERROR;

	if (ModulusSize_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEY_INVALID_MODULUS_PTR_SIZE_ERROR;

	return LLF_RSA_Get_PrivKey(UserPrivKey_ptr, PrivExponent_ptr, PrivExponentSize_ptr, 
		PubExponent_ptr, PubExponentSize_ptr, Modulus_ptr, ModulusSize_ptr);
} /* End of CE2_RSA_Get_PrivKey */

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_PrivKeyCRT
*
* Inputs:
*  @param UserPrivKey_ptr [in] - a pointer to the publick key structure;
*  @param P_ptr [out] - a pointer to the first factor stream of bytes;
*  @param PSize_ptr [in/out] - the size of the first factor buffer in bytes, it is
*         updated to the actual size of the first factor, in bytes;
*  @param Q_ptr [out] - a pointer to the second factor stream of bytes;
*  @param QSize_ptr [in/out] - the size of the second factor buffer in bytes , it is
*         updated to the actual size of the second factor, in bytes;
*  @param dP_ptr [out] - a pointer to the first factors CRT exponent stream of bytes;
*  @param dPSize_ptr [in/out] - the size of the first factor exponent buffer in bytes, it is
*         updated to the actual size of the first factor exponent, in bytes;
*  @param dQ_ptr [out] - a pointer to the second factors CRT exponent stream of bytes;
*  @param dQSize_ptr [in/out] - the size of the second factors CRT exponent buffer in bytes, it is
*         updated to the actual size of the second factors CRT exponent, in bytes;
*  @param qInv_ptr [out] - a pointer to the first CRT coefficient stream of bytes;
*  @param qInvSize_ptr [in/out] - the size of the first CRT coefficient buffer in bytes, it is
*         updated to the actual size of the first CRT coefficient, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Get_PrivKeyCRT gets the p,q,dP,dQ,qP parts of private key
*  from the database.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_RSA_Build_PubKey
*     to get the p,q,dP,dQ,qP parts of private key from the database
*     with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_PrivKeyCRT(
                        CE2_RSAUserPrivKey_t *UserPrivKey_ptr,	/* in */
                        DxUint8_t  *P_ptr,		/* out */
                        DxUint16_t *PSize_ptr,	/* in, out */
                        DxUint8_t  *Q_ptr,		/* out */
                        DxUint16_t *QSize_ptr,	/* in */
                        DxUint8_t  *dP_ptr, 	/* out */
                        DxUint16_t *dPSize_ptr,	/* in, out */
                        DxUint8_t  *dQ_ptr,		/* out */
                        DxUint16_t *dQSize_ptr,	/* in, out */
                        DxUint8_t  *qInv_ptr,	/* out */
                        DxUint16_t *qInvSize_ptr)/* in, out */
{
	if (UserPrivKey_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEYCRT_INVALID_PRIV_KEY_PTR_ERROR;

	if (P_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEYCRT_INVALID_P_PTR_ERROR;

  if (PSize_ptr == DX_NULL)
    return CE2_RSA_GET_PRIVKEYCRT_INVALID_P_SIZE_PTR_ERROR;

  if (Q_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEYCRT_INVALID_Q_PTR_ERROR;

  if (QSize_ptr == DX_NULL)
    return CE2_RSA_GET_PRIVKEYCRT_INVALID_Q_SIZE_PTR_ERROR;

  if (dP_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEYCRT_INVALID_DP_PTR_ERROR;

  if (dPSize_ptr == DX_NULL)
    return CE2_RSA_GET_PRIVKEYCRT_INVALID_DP_SIZE_PTR_ERROR;

	if (dQ_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEYCRT_INVALID_DQ_PTR_ERROR;

  if (dQSize_ptr == DX_NULL)
    return CE2_RSA_GET_PRIVKEYCRT_INVALID_DQ_SIZE_PTR_ERROR;

  if (qInv_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEYCRT_INVALID_QINV_PTR_ERROR;

	if (qInvSize_ptr == DX_NULL)
		return CE2_RSA_GET_PRIVKEYCRT_INVALID_QINV_SIZE_PTR_ERROR;

  if (((CE2RSAPrivKey_t *)UserPrivKey_ptr->PrivateKeyDbBuff)->mode !=
    CE2_RSA_PRIVATE_KEY_CRT_MODE)
    return CE2_RSA_GET_PRIVKEYCRT_IS_NOT_CRT_KEY_ERROR;

	return LLF_RSA_Get_PrivKeyCRT(UserPrivKey_ptr, P_ptr, PSize_ptr, 
		Q_ptr, QSize_ptr, dP_ptr, dPSize_ptr, dQ_ptr, dQSize_ptr,
		qInv_ptr, qInvSize_ptr);
} /* End of CE2_RSA_Get_PrivKeyCRT */

/**
****************************************************************
* Function Name: 
*  CE2_RSA_Get_PrivKeyFull
*
* Inputs:
*  @param UserPrivKeyFull_ptr [in] - a pointer to the publick key structure;
*  @param PrivExponent_ptr [out] - A pointer to the exponent stream of bytes;
*  @param PrivExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param PubExponent_ptr [out] - a pointer to the public exponent stream of bytes;
*  @param PubExponentSize_ptr [in/out] - the size of the exponent buffer in bytes,
*         it is updated to the actual size of the exponent, in bytes;
*  @param Modulus_ptr     [out] - A pointer to the modulus stream of bytes.
*         The MS (most significant) bit must be set to '1'.
*  @param ModulusSize_ptr [in/out] - the size of the modolous buffer in bytes,
*         it is updated to the actual size of the modolous, in bytes.
*  @param P_ptr [out] - a pointer to the first factor stream of bytes;
*  @param PSize_ptr [in/out] - the size of the first factor buffer in bytes, it is
*         updated to the actual size of the first factor, in bytes;
*  @param Q_ptr [out] - a pointer to the second factor stream of bytes;
*  @param QSize_ptr [in/out] - the size of the second factor buffer in bytes , it is
*         updated to the actual size of the second factor, in bytes;
*  @param dP_ptr [out] - a pointer to the first factors CRT exponent stream of bytes;
*  @param dPSize_ptr [in/out] - the size of the first factor exponent buffer in bytes, it is
*         updated to the actual size of the first factor exponent, in bytes;
*  @param dQ_ptr [out] - a pointer to the second factors CRT exponent stream of bytes;
*  @param dQSize_ptr [in/out] - the size of the second factors CRT exponent buffer in bytes, it is
*         updated to the actual size of the second factors CRT exponent, in bytes;
*  @param qInv_ptr [out] - a pointer to the first CRT coefficient stream of bytes;
*  @param qInvSize_ptr [in/out] - the size of the first CRT coefficient buffer in bytes, it is
*         updated to the actual size of the first CRT coefficient, in bytes.
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code.
*
* \brief \b
* Description:
*  CE2_RSA_Get_PrivKeyFull gets the N,e,d,p,q,dP,dQ,qP parts of private key
*  from the database.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call CE2_RSA_Get_PrivKey and CE2_RSA_Get_PrivKeyCRT
*     to get the N,e,d,p,q,dP,dQ,qP parts of private key from the database
*     with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_RSA_Get_PrivKeyFull(
  CE2_RSAUserPrivKey_t  *UserPrivKeyFull_ptr,
  DxUint8_t             *PrivExponent_ptr,
  DxUint16_t            *PrivExponentSize_ptr,
  DxUint8_t             *PubExponent_ptr,
  DxUint16_t            *PubExponentSize_ptr,
  DxUint8_t             *Modulus_ptr,
  DxUint16_t            *ModulusSize_ptr,
  DxUint8_t             *P_ptr, 
  DxUint16_t            *PSize_ptr,
  DxUint8_t             *Q_ptr,
  DxUint16_t            *QSize_ptr,
  DxUint8_t             *dP_ptr, 
  DxUint16_t            *dPSize_ptr,
  DxUint8_t             *dQ_ptr,
  DxUint16_t            *dQSize_ptr,
  DxUint8_t             *qInv_ptr,
  DxUint16_t            *qInvSize_ptr)
{
  CE2Error_t error;

  error = CE2_RSA_Get_PrivKey(UserPrivKeyFull_ptr, PrivExponent_ptr,
     PrivExponentSize_ptr, PubExponent_ptr, PubExponentSize_ptr,
     Modulus_ptr, ModulusSize_ptr);
  if (error != CE2_OK)
    return error;

  return CE2_RSA_Get_PrivKeyCRT(UserPrivKeyFull_ptr, P_ptr, PSize_ptr, Q_ptr, 
    QSize_ptr, dP_ptr, dPSize_ptr, dQ_ptr, dQSize_ptr, qInv_ptr, qInvSize_ptr);
}
